{
  To compare files and directories.

  Copyright (c) 1997 by Wei Ke (AKA KEWEI)

  Author: Wei Ke

  The command line syntax is: XCOMP [-s] [-m [##]] [-q] file1 [file2]

  Options:
  -s      include subdirectories
  -m ##   maximum mismatches per file comparison, default: 10
  -q      display errors only

  file, file2 may be any file path, long file name (quoted if containing
  spaces) and wildcards are allowed, if name portion is missing then "*.*"
  is assumed, if directory portion is missing then current directory is
  assumed.

  To Compile: DCC32.EXE -CC %1
}

program XComp;

uses SysUtils, Windows;

const
  BUFFERLENGTH = 64000;
  FILEFATTR = faArchive or faHidden or faSysFile or faReadOnly;
  DIRFATTR = faHidden or faDirectory;

type
  TFile = class(TObject)
  public
    sName: String;
    fd: Integer;

    constructor Create(const s: String);
    destructor Destroy; override;

    function GetSize: Integer;
    procedure Open;
    procedure Close;
    procedure Read(var buffer; n: Integer);
  end;

  EFileError = class(Exception)
  public
    ef: TFile;
    ec: Integer;
    em: String;

    constructor Create(f: TFile; c: Integer; const m: String);
  end;

  ECompError = class(Exception)
  end;

  TBuffer = array [0..BUFFERLENGTH-1] of Byte;

var
  buffer1, buffer2: TBuffer;
  sSrcBaseDir, sDestBaseDir, sRelDir: String;
  sSrcName, sDestName, sDestExt: String;
  nMaxDiff: Integer;
  boIncludeSub, boQuiet: Boolean;
  nTotalComp, nTotalOK, nTotalIOErr, nTotalDiffSize: Integer;

constructor TFile.Create(const s: String);
begin
  inherited Create;
  sName := s;
  fd := -1
end;

destructor TFile.Destroy;
begin
  Close;
  inherited Destroy
end;

procedure TFile.Open;
begin
  Close;
  fd := FileOpen(sName, fmOpenRead or fmShareDenyWrite);
  if fd < 0 then
    raise EFileError.Create(self, GetLastError, 'FileOpen');
end;

procedure TFile.Close;
begin
  if fd >= 0 then begin
    FileClose(fd);
    fd := -1;
  end
end;

function TFile.GetSize: Integer;
var
  p, l: Integer;
begin
  p := FileSeek(fd, 0, 1);
  if p < 0 then
    raise EFileError.Create(self, GetLastError, 'FileSeek');
  l := FileSeek(fd, 0, 2);
  if l < 0 then
    raise EFileError.Create(self, GetLastError, 'FileSeek');
  result := l;
  l := FileSeek(fd, 0, p);
  if l <> p then
    raise EFileError.Create(self, GetLastError, 'FileSeek');
end;

procedure TFile.Read(var buffer; n: Integer);
var
  l: Integer;
begin
  l := FileRead(fd, buffer, n);
  if l <> n then
    raise EFileError.Create(self, GetLastError, 'FileRead');
end;

constructor EFileError.Create(f: TFile; c: Integer; const m: String);
begin
  inherited Create('');
  ef := f;
  ec := c;
  em := m
end;

function ExtractFileNameNoExt(const s: String): String;
var
  sName: String;
  lExt: Integer;
begin
  sName := ExtractFileName(s);
  lExt := Length(ExtractFileExt(s));
  result := Copy(sName, 1, Length(sName)-lExt)
end;

procedure GenCorresponding(var sResult: String; sGiven, sPattern: String);
var
  iG, iP: Integer;
begin
  iG := 1;
  iP := 1;
  sResult := '';
  while (iG <= Length(sGiven)) and (iP <= Length(sPattern)) do begin
    case sPattern[iP] of
      '?': begin
        sResult := sResult+sGiven[iG];
        Inc(iP)
      end;
      '*': begin
        sResult := sResult+sGiven[iG];
      end;
    else
      sResult := sResult+sPattern[iP];
      Inc(iP)
    end;
    Inc(iG);
  end;

  while iP <= Length(sPattern) do begin
    if not (sPattern[iP] in ['*', '?']) then
      sResult := sResult+sPattern[iP];
    Inc(iP)
  end
end;

procedure CompFiles(const s1, s2: String);
var
  f1, f2: TFile;
  p, nc, l, l1, l2: Integer;
  sComp: String;

  procedure DisplayComp;
  begin
    if sComp <> '' then begin
      Writeln(sComp);
      sComp := '';
    end
  end;

  procedure CompBlocks(const b1, b2: TBuffer; p, l: Integer; var n: Integer);
  var
    i: Integer;
  begin
    for i := 0 to l-1 do begin
      if b1[i] <> b2[i] then begin
        DisplayComp;
        Writeln('>>>> ', IntToHex(p+i, 8), ': ', IntToHex(b1[i], 2), ' ', IntToHex(b2[i], 2));
        Dec(n);
        if n <= 0 then
          raise ECompError.Create('')
      end
    end
  end;

begin
  sComp := '"'+s1+'" ?= "'+s2+'"';
  if not boQuiet then
    DisplayComp;
  Inc(nTotalComp);
  f1 := TFile.Create(s1);
  f2 := TFile.Create(s2);
  try
    f1.Open;
    f2.Open;

    l1 := f1.GetSize;
    l2 := f2.GetSize;

    if l1 <> l2 then begin
      DisplayComp;
      Writeln('>>>> file sizes are differrent: ', l1, ' ', l2);
      Inc(nTotalDiffSize)
    end;

    if l1 < l2 then
      l := l1
    else
      l := l2;

    nc := nMaxDiff;

    p := 0;

    while l > BUFFERLENGTH do begin
      f1.Read(buffer1, BUFFERLENGTH);
      f2.Read(buffer2, BUFFERLENGTH);
      CompBlocks(buffer1, buffer2, p, BUFFERLENGTH, nc);
      Dec(l, BUFFERLENGTH);
      Inc(p, BUFFERLENGTH);
    end;

    if l > 0 then begin
      f1.Read(buffer1, l);
      f2.Read(buffer2, l);
      CompBlocks(buffer1, buffer2, p, l, nc);
    end;

    f1.Close;
    f2.Close;

    if nc = nMaxDiff then
      Inc(nTotalOK);

  except
    on e: EFileError do begin
      DisplayComp;
      Writeln('>>>> "', e.ef.sName, '" ', e.em, ': ', e.ec, ': ', SysErrorMessage(e.ec));
      f1.Close;
      f2.Close;
      Inc(nTotalIOErr)
    end;
    on ECompError do begin
      f1.Close;
      f2.Close
    end
  end
end;

procedure FoundF(const s: String);
var
  sName: String;
  sExt: String;
begin
  GenCorresponding(sName, ExtractFileNameNoExt(s), sDestName);
  GenCorresponding(sExt, ExtractFileExt(s), sDestExt);
  if sExt = '.' then
    sExt := '';
  CompFiles(sSrcBaseDir+sRelDir+s, sDestBaseDir+sRelDir+sName+sExt);
  SetConsoleTitle(PChar(IntToStr(nTotalComp)+','+IntToStr(nTotalOK)+' - XComp'))
end;

procedure SearchF;
var
  sr: TSearchRec;
  ec, nMark: Integer;
begin
  ec := SysUtils.FindFirst(sSrcBaseDir+sRelDir+sSrcName, FILEFATTR, sr);
  while ec = 0 do begin
    FoundF(sr.Name);
    ec := SysUtils.FindNext(sr)
  end;
  SysUtils.FindClose(sr);

  if boIncludeSub then begin
    ec := SysUtils.FindFirst(sSrcBaseDir+sRelDir+'*.*', DIRFATTR, sr);
    while ec = 0 do begin
      if ((sr.Attr or faDirectory) <> 0) and (sr.Name <> '.') and (sr.Name <> '..') then begin
        nMark := Length(sRelDir);
        sRelDir := sRelDir+sr.Name+'\';
        SearchF;
        Delete(sRelDir, nMark+1, Length(sRelDir)-nMark)
      end;
      ec := SysUtils.FindNext(sr)
    end;
    SysUtils.FindClose(sr)
  end
end;

function IsDirectory(const s: String): Boolean;
var
  sr: TSearchRec;
begin
  result := (Pos('*', s) = 0) and (Pos('?', s) = 0);
  if result then begin
    result := result and (SysUtils.FindFirst(s, DIRFATTR, sr) = 0) and ((sr.Attr and faDirectory) <> 0);
    SysUtils.FindClose(sr)
  end
end;

procedure ParseFileNames(const s1, s2: String);
var
  s: String;
begin
  if Length(ExtractFileName(s1)) = 0 then
    s := ExtractFilePath(s1)+'*.*'
  else if IsDirectory(s1) then
    s := s1+'\*.*'
  else
    s := s1;
  sSrcBaseDir := ExtractFilePath(ExpandFileName(s));
  sSrcName := ExtractFileName(s);

  if Length(ExtractFileName(s2)) = 0 then
    s := ExtractFilePath(s2)+'*.*'
  else if IsDirectory(s2) then
    s := s2+'\*.*'
  else
    s := s2;

  sDestBaseDir := ExtractFilePath(ExpandFileName(s));
  sDestName := ExtractFileNameNoExt(s);
  sDestExt := ExtractFileExt(s);
end;

procedure ParseCommandLine;
var
  vs: array [0..1] of String;
  i, j, n: Integer;
begin
  i := 1;
  j := 0;
  vs[0] := '';
  vs[1] := '';
  while i <= ParamCount do begin
    if UpperCase(ParamStr(i)) = '-S' then
      boIncludeSub := TRUE
    else if UpperCase(ParamStr(i)) = '-Q' then
      boQuiet := TRUE
    else if UpperCase(ParamStr(i)) = '-M' then begin
      try
        n := StrToInt(ParamStr(i+1));
        Inc(i);
        if n > 0 then
          nMaxDiff := n;
      except
        on EConvertError do ;
      end
    end else if j < 2 then begin
      vs[j] := ParamStr(i);
      Inc(j)
    end;
    Inc(i)
  end;
  ParseFileNames(vs[0], vs[1])
end;

begin
  if ParamCount = 0 then begin
    Writeln('XCOMP Utility V1.3 Copyright (c) 1997, 1998 by Wei Ke, All rights reserved');
    Writeln('Usage: XCOMP [-s] [-m [##]] [-q] file1 [file2]');
    Writeln('  -s      include subdirectories');
    Writeln('  -m ##   maximum mismatches per file comparison, default: 10');
    Writeln('  -q      display errors only');
  end else begin
    nMaxDiff := 10;
    boIncludeSub := FALSE;
    boQuiet := FALSE;
    ParseCommandLine;
    sRelDir := '';
    nTotalComp := 0;
    nTotalOK := 0;
    nTotalIOErr := 0;
    nTotalDiffSize := 0;
    SearchF;
    Write(nTotalComp, ' comparison(s), ', nTotalOK, ' OK');
    if nTotalDiffSize <> 0 then
      Write(', ', nTotalDiffSize, ' of different sizes');
    if nTotalIOErr <> 0 then
      Write(', ', nTotalIOErr, ' file I/O error(s)');
    Writeln;
  end
end.